<?php
/**
 * Webhook para Recargas via Pix (PushinPay)
 * Recebe a notificação de pagamento e atualiza o status da recarga e o saldo do cliente.
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
date_default_timezone_set('America/Sao_Paulo'); 

// Caminhos dos arquivos JSON
$base_path = __DIR__ . '/data/';
$clientes_file  = $base_path . 'clientes.json';
$recargas_file  = $base_path . 'recargas.json';
$log_file       = $base_path . 'webhook_recarga_log.txt'; // Arquivo de log para debug

// Funções de manipulação de JSON
function load_json($file) {
    return file_exists($file) ? json_decode(file_get_contents($file), true) ?: [] : [];
}

function save_json($file, $data) {
    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function log_message($message) {
    global $log_file;
    file_put_contents($log_file, date('Y-m-d H:i:s') . " - " . $message . "\n", FILE_APPEND);
}

// 1. Receber e decodificar o payload JSON
$payload = file_get_contents('php://input');
$data = json_decode($payload, true);

log_message("Webhook recebido. Payload: " . $payload);

// 2. Verificar se o payload é válido e se a transação foi paga/aprovada
if (empty($data) || !isset($data['id']) || !isset($data['status'])) {
    log_message("Payload inválido ou incompleto.");
    http_response_code(400);
    exit;
}

$transaction_id = $data['id'];
$status = $data['status'];

// Apenas processar se o status for de sucesso
if ($status !== 'paid' && $status !== 'approved') {
    log_message("Status da transação não é de pagamento: " . $status);
    http_response_code(200); // Retorna 200 para indicar que a notificação foi recebida
    exit;
}

// 3. Carregar dados
$recargas = load_json($recargas_file);
$clientes = load_json($clientes_file);
$recarga_encontrada = false;
$cliente_encontrado = false;
$mudou = false;

// 4. Buscar a recarga correspondente
foreach ($recargas as $key => &$r) {
    if (($r['transaction_id'] ?? '') === $transaction_id) {
        $recarga_encontrada = true;
        
        // 5. Verificar se a recarga já foi paga e creditada
        if (($r['status_pagamento'] ?? '') === 'pago' && ($r['creditado'] ?? false)) {
            log_message("Recarga {$r['id']} já estava paga e creditada. Nenhuma ação necessária.");
            http_response_code(200);
            exit;
        }
        
        // 6. Atualizar status da recarga
        $r['status_pagamento'] = 'pago';
        $r['pago_em'] = date('c');
        $idCliente = $r['id_cliente'];
        $valorRecarga = (float)$r['valor'];
        $mudou = true;

        log_message("Recarga {$r['id']} encontrada. Status atualizado para 'pago'.");

        // 7. Creditar o saldo ao cliente
        foreach ($clientes as $c_key => &$c) {
            if ($c['id'] === $idCliente) {
                $cliente_encontrado = true;
                
                // Evitar crédito duplicado (garantia extra)
                if (!($r['creditado'] ?? false)) {
                    $c['saldo'] = (float)($c['saldo'] ?? 0) + $valorRecarga;
                    $r['creditado'] = true;
                    log_message("Saldo de R$ {$valorRecarga} creditado ao cliente {$c['id']}. Novo saldo: R$ {$c['saldo']}");
                } else {
                    log_message("Tentativa de crédito duplicado para recarga {$r['id']}. Ação bloqueada.");
                }
                break;
            }
        }
        
        if (!$cliente_encontrado) {
            log_message("ERRO: Cliente {$idCliente} não encontrado para a recarga {$r['id']}.");
        }
        
        break;
    }
}

if (!$recarga_encontrada) {
    log_message("ERRO: Recarga com transaction_id {$transaction_id} não encontrada.");
    http_response_code(404);
    exit;
}

// 8. Salvar alterações nos arquivos JSON
if ($mudou) {
    save_json($recargas_file, $recargas);
    save_json($clientes_file, $clientes);
    log_message("Arquivos JSON salvos com sucesso.");
}

// 9. Resposta de sucesso para o webhook
http_response_code(200);
echo json_encode(['success' => true, 'message' => 'Webhook processado com sucesso.']);

?>
