<?php
/**
 * Webhook Profissional PushinPay
 * Processamento em tempo real de Pedidos e Recargas
 */

// Configurações de Segurança e Erros
error_reporting(E_ALL);
ini_set('display_errors', 0);
date_default_timezone_set('America/Sao_Paulo');

$base_path = __DIR__ . '/data/';
$log_file = $base_path . 'webhook_pushinpay.log';

// Função de Log Profissional
function logger($message, $data = null) {
    global $log_file;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] $message" . ($data ? " | Data: " . json_encode($data) : "") . PHP_EOL;
    file_put_contents($log_file, $logEntry, FILE_APPEND);
}

// 1. Captura o Payload
$payload = file_get_contents('php://input');
$data = json_decode($payload, true);

if (!$data) {
    logger("ERRO: Payload inválido ou vazio recebido.");
    http_response_code(400);
    exit('Payload Inválido');
}

logger("INFO: Webhook recebido", $data);

// 2. Validação de Status (Apenas 'paid' nos interessa para liberar)
if (!isset($data['status']) || $data['status'] !== 'paid') {
    logger("INFO: Status ignorado: " . ($data['status'] ?? 'N/A'));
    http_response_code(200); // Respondemos 200 para a API não reenviar
    exit('Status não processável');
}

$externalId = $data['external_id'] ?? '';
$pushinpayId = $data['id'] ?? ''; // ID da transação na PushinPay

if (empty($externalId)) {
    logger("ERRO: external_id não encontrado no payload.");
    http_response_code(200);
    exit('Sem external_id');
}

// 3. Carregamento de Dados
$pedidos_file = $base_path . 'pedidos.json';
$clientes_file = $base_path . 'clientes.json';
$recargas_file = $base_path . 'recargas.json';

function get_json($file) { return file_exists($file) ? json_decode(file_get_contents($file), true) : []; }
function set_json($file, $data) { file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)); }

$atualizado = false;

// 4. Processamento de Pedidos (Produtos)
if (strpos($externalId, 'p') === 0) {
    $pedidos = get_json($pedidos_file);
    foreach ($pedidos as &$p) {
        if ($p['id'] === $externalId) {
            if ($p['status_pagamento'] !== 'pago') {
                $p['status_pagamento'] = 'pago';
                $p['pushinpay_id'] = $pushinpayId;
                $p['pago_em'] = date('c');
                $atualizado = true;
                set_json($pedidos_file, $pedidos);
                logger("SUCESSO: Pedido $externalId aprovado.");
            } else {
                logger("AVISO: Pedido $externalId já estava pago.");
                $atualizado = true;
            }
            break;
        }
    }
}

// 5. Processamento de Recargas (Saldo)
if (!$atualizado && strpos($externalId, 'rec') === 0) {
    $recargas = get_json($recargas_file);
    $clientes = get_json($clientes_file);
    $idCliente = null;
    $valorRecarga = 0;

    foreach ($recargas as &$r) {
        if ($r['id'] === $externalId) {
            if ($r['status'] !== 'pago') {
                $r['status'] = 'pago';
                $r['pushinpay_id'] = $pushinpayId;
                $r['pago_em'] = date('c');
                $idCliente = $r['id_cliente'];
                $valorRecarga = (float)$r['valor'];
                $atualizado = true;
                set_json($recargas_file, $recargas);
                logger("SUCESSO: Recarga $externalId aprovada. Valor: $valorRecarga");
            } else {
                logger("AVISO: Recarga $externalId já estava paga.");
                $atualizado = true;
            }
            break;
        }
    }

    if ($atualizado && $idCliente && $valorRecarga > 0) {
        $clienteEncontrado = false;
        foreach ($clientes as &$c) {
            if ($c['id'] === $idCliente) {
                $c['saldo'] = (float)($c['saldo'] ?? 0) + $valorRecarga;
                $clienteEncontrado = true;
                logger("SUCESSO: Saldo do cliente $idCliente atualizado. Novo saldo: " . $c['saldo']);
                break;
            }
        }
        if ($clienteEncontrado) {
            set_json($clientes_file, $clientes);
        }
    }
}

// 6. Resposta Final
if ($atualizado) {
    http_response_code(200);
    echo json_encode(['status' => 'success', 'message' => 'Pagamento processado']);
} else {
    logger("ERRO: Transação $externalId não encontrada nos registros locais.");
    http_response_code(200);
    echo json_encode(['status' => 'not_found', 'message' => 'Registro não localizado']);
}
?>
