<?php
/**
 * Página de Histórico de Pagamentos - pagamentos.php (Versão Refinada com Métodos e Múltiplos Produtos)
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);

$idParceiro = isset($_GET['id']) ? (string)$_GET['id'] : '';
$base_path = __DIR__ . '/data/';
$parceiros_file = $base_path . 'parceiros.json';
$pedidos_file   = $base_path . 'pedidos.json';
$recargas_file  = $base_path . 'recargas.json';

function load_json_data($file) {
    if (!file_exists($file)) return [];
    $content = file_get_contents($file);
    return json_decode($content, true) ?: [];
}

$parceiros = load_json_data($parceiros_file);
$parceiro = null;
foreach($parceiros as $p){
    if((string)($p['id'] ?? '') === $idParceiro){
        $parceiro = $p;
        break;
    }
}

if(!$parceiro) {
    $parceiro = ['nome' => 'Loja Parceira', 'cor_primaria' => '#2563eb'];
}

// API Interna para carregar transações com Paginação
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'get_pagamentos') {
    header('Content-Type: application/json');
    $idCliente = $_POST['id_cliente'];
    $page = isset($_POST['page']) ? (int)$_POST['page'] : 1;
    $perPage = 10;

    $pedidos = load_json_data($pedidos_file);
    $recargas = load_json_data($recargas_file);

    $transacoes = [];
    
    // Adicionar Compras com detalhamento de produtos e métodos
    foreach ($pedidos as $p) {
        if ($p['id_cliente'] === $idCliente) {
            $nomesProdutos = [];
            
            // Extração de nomes do array 'arquivos'
            if(isset($p['arquivos']) && is_array($p['arquivos'])) {
                foreach($p['arquivos'] as $arq) {
                    if(isset($arq['nome'])) $nomesProdutos[] = $arq['nome'];
                }
            } 
            // Fallback para 'itens'
            elseif(isset($p['itens']) && is_array($p['itens'])) {
                foreach($p['itens'] as $item) {
                    if(isset($item['nome'])) $nomesProdutos[] = $item['nome'];
                }
            }

            $descricao = !empty($nomesProdutos) ? implode(", ", $nomesProdutos) : "Compra de Produto";

            $transacoes[] = [
                "id" => $p['id'],
                "tipo" => "Compra",
                "descricao" => $descricao,
                "valor" => $p['valor_total'],
                "status" => $p['status_pagamento'],
                "metodo" => $p['metodo'] ?? 'N/A',
                "data" => $p['data'],
                "natureza" => "saida"
            ];
        }
    }

    // Adicionar Recargas
    foreach ($recargas as $r) {
        if ($r['id_cliente'] === $idCliente) {
            $transacoes[] = [
                "id" => $r['id'],
                "tipo" => "Recarga",
                "descricao" => "Adição de Saldo em Conta",
                "valor" => $r['valor'],
                "status" => $r['status'],
                "metodo" => "pix",
                "data" => $r['data'],
                "natureza" => "entrada"
            ];
        }
    }

    // Ordenar por data (mais recente primeiro)
    usort($transacoes, function($a, $b) {
        return strtotime($b['data']) - strtotime($a['data']);
    });

    $totalItems = count($transacoes);
    $totalPages = ceil($totalItems / $perPage);
    $offset = ($page - 1) * $perPage;
    $pagedData = array_slice($transacoes, $offset, $perPage);

    echo json_encode([
        "success" => true, 
        "transacoes" => $pagedData,
        "pagination" => [
            "current_page" => $page,
            "total_pages" => $totalPages,
            "total_items" => $totalItems
        ]
    ]);
    exit;
}
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Pagamentos | <?= htmlspecialchars($parceiro['nome']) ?></title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700;800;900&display=swap" rel="stylesheet">
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://unpkg.com/@phosphor-icons/web"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: { primary: '<?= $parceiro['cor_primaria'] ?? "#2563eb" ?>' },
                    fontFamily: { sans: ['Inter', 'sans-serif'] }
                }
            }
        }
    </script>
    <style>
        :root { --p-color: <?= $parceiro['cor_primaria'] ?? "#2563eb" ?>; }
        body { background-color: #f8fafc; color: #1e293b; }
        header { background: rgba(255, 255, 255, 0.9); backdrop-filter: blur(10px); border-bottom: 1px solid #e2e8f0; position: sticky; top: 0; z-index: 100; }
        .btn-primary { background-color: var(--p-color); color: #fff !important; transition: all 0.3s; }
        .status-badge { padding: 4px 10px; border-radius: 8px; font-size: 10px; font-weight: 800; text-transform: uppercase; letter-spacing: 0.5px; }
        .status-pago { background: #dcfce7; color: #166534; }
        .status-pendente { background: #fef9c3; color: #854d0e; }
        .natureza-entrada { color: #16a34a; }
        .natureza-saida { color: #dc2626; }
        .card-table { background: #fff; border: 1px solid #e2e8f0; border-radius: 24px; overflow: hidden; }
        .pagination-btn { padding: 8px 16px; border-radius: 12px; font-size: 12px; font-weight: 700; transition: all 0.2s; border: 1px solid #e2e8f0; }
        .pagination-btn:hover:not(:disabled) { border-color: var(--p-color); color: var(--p-color); }
        .pagination-btn:disabled { opacity: 0.3; cursor: not-allowed; }
        .pagination-btn.active { background: var(--p-color); color: #fff; border-color: var(--p-color); }
        .metodo-tag { font-size: 9px; font-weight: 900; padding: 2px 6px; border-radius: 4px; background: #f1f5f9; color: #64748b; text-transform: uppercase; }
    </style>
</head>
<body class="min-h-screen flex flex-col">

<?php include 'includes/menu.php'; ?>

<main class="max-w-5xl mx-auto px-4 py-12 w-full">
    <div id="authCheck" class="hidden">
        <div class="bg-white rounded-3xl shadow-sm border border-slate-100 p-16 text-center">
            <i class="ph ph-lock-key text-6xl text-slate-200 mb-6"></i>
            <h2 class="text-2xl font-black text-slate-800 mb-2 uppercase">Acesso Restrito</h2>
            <p class="text-slate-500 text-sm mb-8">Faça login para ver seu histórico de pagamentos.</p>
            <a href="carrinho.php?id=<?= $idParceiro ?>" class="btn-primary px-10 py-4 rounded-2xl font-bold text-xs uppercase tracking-widest inline-block">Entrar na Conta</a>
        </div>
    </div>

    <div id="contentView" class="hidden space-y-8">
        <div class="flex flex-col md:flex-row md:items-end justify-between gap-6">
            <div>
                <h1 class="text-4xl font-black text-slate-900 tracking-tighter uppercase italic leading-none">Pagamentos</h1>
                <p class="text-slate-500 text-sm mt-2">Histórico detalhado de suas transações.</p>
            </div>
            <a href="conta.php?id=<?= $idParceiro ?>" class="text-[10px] font-black text-slate-400 border border-slate-200 px-4 py-2 rounded-xl hover:bg-slate-50 transition-all uppercase tracking-widest text-center">Voltar ao Painel</a>
        </div>

        <!-- Tabela para Desktop -->
        <div class="hidden md:block card-table shadow-sm">
            <table class="w-full text-left border-collapse">
                <thead>
                    <tr class="bg-slate-50/50">
                        <th class="p-5 text-[10px] font-black text-slate-400 uppercase tracking-widest">Data</th>
                        <th class="p-5 text-[10px] font-black text-slate-400 uppercase tracking-widest">Produtos / Descrição</th>
                        <th class="p-5 text-[10px] font-black text-slate-400 uppercase tracking-widest">Método</th>
                        <th class="p-5 text-[10px] font-black text-slate-400 uppercase tracking-widest">Valor</th>
                        <th class="p-5 text-[10px] font-black text-slate-400 uppercase tracking-widest">Status</th>
                    </tr>
                </thead>
                <tbody id="pagamentosTable">
                    <!-- JS preenche aqui -->
                </tbody>
            </table>
        </div>

        <!-- Lista para Mobile -->
        <div id="pagamentosListMobile" class="md:hidden space-y-4">
            <!-- JS preenche aqui -->
        </div>

        <!-- Paginação -->
        <div class="flex items-center justify-between bg-white p-4 rounded-2xl border border-slate-100">
            <p class="text-[10px] font-bold text-slate-400 uppercase tracking-widest">
                Mostrando <span id="countItems" class="text-slate-900">0</span> transações
            </p>
            <div class="flex gap-2" id="paginationControls">
                <!-- JS preenche aqui -->
            </div>
        </div>
    </div>
</main>

<script>
    const idParceiro = '<?= $idParceiro ?>';
    const cliente = JSON.parse(localStorage.getItem('cliente_logado_' + idParceiro));
    let currentPage = 1;

    async function carregarPagamentos(page = 1) {
        if (!cliente) return;
        currentPage = page;

        const fd = new FormData();
        fd.append('action', 'get_pagamentos');
        fd.append('id_cliente', cliente.id);
        fd.append('page', page);

        const res = await fetch(window.location.href, { method: 'POST', body: fd });
        const data = await res.json();

        if (data.success) {
            // Desktop Table
            const table = document.getElementById('pagamentosTable');
            table.innerHTML = data.transacoes.map(t => `
                <tr class="border-t border-slate-50 hover:bg-slate-50/30 transition-colors">
                    <td class="p-5 text-xs font-medium text-slate-500">
                        ${new Date(t.data).toLocaleDateString('pt-BR')}
                        <span class="block text-[9px] opacity-50">${new Date(t.data).toLocaleTimeString('pt-BR', {hour: '2-digit', minute:'2-digit'})}</span>
                    </td>
                    <td class="p-5">
                        <p class="text-[10px] font-black text-primary uppercase tracking-tighter mb-1">${t.tipo}</p>
                        <p class="text-xs font-bold text-slate-700 leading-tight max-w-xs">${t.descricao}</p>
                        <p class="text-[9px] text-slate-300 font-mono mt-1">ID: ${t.id}</p>
                    </td>
                    <td class="p-5">
                        <span class="metodo-tag">${t.metodo}</span>
                    </td>
                    <td class="p-5 text-sm font-black ${t.natureza === 'entrada' ? 'natureza-entrada' : 'natureza-saida'}">
                        ${t.natureza === 'entrada' ? '+' : '-'} ${new Intl.NumberFormat('pt-BR', { style: 'currency', currency: 'BRL' }).format(t.valor)}
                    </td>
                    <td class="p-5">
                        <span class="status-badge status-${t.status}">${t.status}</span>
                    </td>
                </tr>
            `).join('') || '<tr><td colspan="5" class="p-20 text-center text-slate-400 text-xs font-bold uppercase tracking-widest">Nenhum registro encontrado</td></tr>';

            // Mobile List
            const listMobile = document.getElementById('pagamentosListMobile');
            listMobile.innerHTML = data.transacoes.map(t => `
                <div class="bg-white p-5 rounded-2xl border border-slate-100 space-y-3">
                    <div class="flex justify-between items-start">
                        <div>
                            <p class="text-[9px] font-black text-primary uppercase tracking-tighter">${t.tipo} <span class="ml-2 metodo-tag">${t.metodo}</span></p>
                            <p class="text-xs font-bold text-slate-800 mt-1">${t.descricao}</p>
                        </div>
                        <span class="status-badge status-${t.status}">${t.status}</span>
                    </div>
                    <div class="flex justify-between items-end pt-2 border-t border-slate-50">
                        <div class="text-[9px] text-slate-400 font-medium">
                            ${new Date(t.data).toLocaleDateString('pt-BR')} às ${new Date(t.data).toLocaleTimeString('pt-BR', {hour: '2-digit', minute:'2-digit'})}
                        </div>
                        <div class="text-sm font-black ${t.natureza === 'entrada' ? 'natureza-entrada' : 'natureza-saida'}">
                            ${t.natureza === 'entrada' ? '+' : '-'} ${new Intl.NumberFormat('pt-BR', { style: 'currency', currency: 'BRL' }).format(t.valor)}
                        </div>
                    </div>
                </div>
            `).join('');

            // Paginação
            document.getElementById('countItems').textContent = data.pagination.total_items;
            const controls = document.getElementById('paginationControls');
            let html = `
                <button onclick="carregarPagamentos(${currentPage - 1})" ${currentPage === 1 ? 'disabled' : ''} class="pagination-btn">
                    <i class="ph ph-caret-left"></i>
                </button>
            `;
            
            for (let i = 1; i <= data.pagination.total_pages; i++) {
                if (i === 1 || i === data.pagination.total_pages || (i >= currentPage - 1 && i <= currentPage + 1)) {
                    html += `<button onclick="carregarPagamentos(${i})" class="pagination-btn ${i === currentPage ? 'active' : ''}">${i}</button>`;
                } else if (i === currentPage - 2 || i === currentPage + 2) {
                    html += `<span class="px-2 text-slate-300">...</span>`;
                }
            }

            html += `
                <button onclick="carregarPagamentos(${currentPage + 1})" ${currentPage === data.pagination.total_pages ? 'disabled' : ''} class="pagination-btn">
                    <i class="ph ph-caret-right"></i>
                </button>
            `;
            controls.innerHTML = html;
            window.scrollTo({ top: 0, behavior: 'smooth' });
        }
    }

    if (!cliente) {
        document.getElementById('authCheck').classList.remove('hidden');
    } else {
        document.getElementById('contentView').classList.remove('hidden');
        carregarPagamentos();
    }
</script>

</body>
</html>
